<?php
/**
 * Rsssl_Captcha_Protected.php
 *
 * The Rsssl_Captcha_Protected.php file is part of the Really Simple SSL Pro plugin.
 * This file contains the Rsssl_Captcha_Provider class, which is responsible for generating and validating CAPTCHA.
 * The class implements the Rsssl_Captcha_Protected interface and uses functionality from the Rsssl_Captcha_Trait.
 *
 * @file        Rsssl_Captcha_Protected.php
 * @package     REALLY_SIMPLE_SSL
 * @category    Security
 * @subpackage  WordPress\Captcha
 * @author     Marcel Santing <marcel@really-simple-ssl.com>
 * @copyright  2024 Really Simple plugins
 * @version    Release 7.3
 */

namespace REALLY_SIMPLE_SSL\Security\WordPress\Captcha;

/**
 * Class Rsssl_Captcha_Provider
 *
 * The Rsssl_Captcha_Provider class is a part of the Really Simple SSL Pro's security module.
 * It uses the Rsssl_Captcha_Trait to provide methods for display and validation of CAPTCHA.
 * As per the Rsssl_Captcha_Protected interface, this class should provide implementations for checking
 * captcha threshold and validating captcha inputs.
 *
 * @package     RSSSL_PRO\Security\WordPress\Captcha
 * @author     Marcel Santing <marcel@really-simple-ssl.com>
 * @version    PHP 7.2
 * @since      File available since Release 7.3
 */
abstract class Rsssl_Captcha_Provider {

	/**
	 * Displays a captcha image.
	 *
	 * This method generates a captcha image and displays it on the screen.
	 * The generated captcha image is used to prevent automated activities,
	 * such as spamming or brute-force attacks, by requiring users to read
	 * and input the characters displayed in the captcha image.
	 *
	 * Function is available when the trait is used.
	 *
	 * @return void
	 */
	abstract public function render(): void;

	/**
	 * Validates the given response.
	 *
	 * This method is an abstract method which must be implemented by child classes.
	 * It takes in a response and performs the necessary validations to determine if
	 * the response is valid or not. The specific validation logic will vary depending
	 * on the implementation in the child classes.
	 *
	 * @param  mixed $response  The response to be validated.
	 *
	 * @return bool True if the response is valid, False otherwise.
	 */
	abstract public function validate( $response ): bool;

	/**
	 * Retrieves the response field from the captcha verification API.
	 *
	 * This method retrieves the response field from the captcha verification API
	 * to check if the user's input is valid. The response field contains a boolean value
	 * indicating whether the user's captcha input was correct.
	 *
	 * Note that this method is abstract and should be implemented in the class that uses it.
	 * The implementation will depend on the specific captcha verification API being used.
	 *
	 * @return bool The response field value indicating whether the user's captcha input was correct.
	 */
	abstract public function get_response_field(): bool;

	/**
	 * Retrieves the response value.
	 *
	 * This method returns the response value generated by the system.
	 * The response value can be used to determine the outcome of an action,
	 * such as the success or failure of a request.
	 *
	 * Function is available when the trait is used.
	 *
	 * @return string The response value generated by the system.
	 */
	abstract public function get_response_value(): string;

	/**
	 * Returns the configuration settings for the application.
	 *
	 * This method retrieves the configuration settings for the application
	 * and returns them as an array. The configuration settings contain various
	 * values that control the behavior and settings of the application.
	 *
	 * The returned configuration array may include settings like database
	 * connection details, API keys, application specific settings, etc.
	 *
	 * @return array The configuration settings for the application.
	 */
	abstract public function get_config(): array;

	/**
	 * Retrieves the value of a specified key from the $_POST super global.
	 *
	 * This method checks if the specified key exists in the $_POST super global.
	 * If the key exists, it retrieves the value and applies sanitization functions
	 * to ensure the security and integrity of the data.
	 *
	 * @param  string $key  The key for which to retrieve the value from $_POST.
	 *
	 * @return string The sanitized value of the specified key from $_POST, or an
	 * empty string if the key does not exist.
	 */
	public function get_post_value( string $key ): string {
		// phpcs:ignore WordPress.Security
		return isset( $_POST[ $key ] ) ? sanitize_text_field( stripslashes( wp_unslash( $_POST[ $key ] ) ) ) : '';
	}

	/**
	 * Verifies the WordPress nonce.
	 *
	 * This method accepts a nonce key and verifies it by returning a boolean result.
	 *
	 * @return bool Indicates if the nonce is valid.
	 */
	public function verify_nonce(): bool {
		$nonce = $this->get_post_value( 'rsssl_captcha_nonce' );
		return wp_verify_nonce( $nonce, 'rsssl_captcha_nonce' );
	}
}
