jQuery(document).ready(function ($) {
    $(document).on('change input', '.mwpl_range', function (e) {
        let el = $(this),
            valueWrapper = el.closest('.option_field').find('.range_value');
        valueWrapper.text(`( ${el.val()} )`);
    });

    const modalOverlay = $('#negarara-modal-overlay');
    const modalMessage = $('#negarara-modal-message');
    const confirmBtn = $('#negarara-modal-confirm');
    const cancelBtn = $('#negarara-modal-cancel');

    function showConfirmationModal(message, onConfirm) {
        modalMessage.text(message);
        modalOverlay.fadeIn(200);

        confirmBtn.off('click').one('click', function() {
            modalOverlay.fadeOut(200);
            onConfirm();
        });

        cancelBtn.off('click').one('click', function() {
            modalOverlay.fadeOut(200);
        });
    }

    let isConverting = false;
    let totalImages = 0;
    let convertedImages = 0;
    let logMessages = [];
    const logContainer = $('#negarara_conversion_log');
    const progressContainer = $('#negarara_conversion_progress');

    function startBulkConversion() {
        isConverting = true;
        $('#negarara_bulk_convert').prop('disabled', true).text('Converting...');
        progressContainer.html('<p>Preparing to convert images...</p>');
        logContainer.html('').show();
        logMessages = [];
        convertedImages = 0;

        $.ajax({
            url: negararaAjax.ajaxurl,
            type: 'POST',
            data: {
                action: 'negarara_get_convertible_image_count',
                security: negararaAjax.security
            },
            success: function(response) {
                if (response.success) {
                    totalImages = response.data.count;
                    if (totalImages > 0) {
                        logMessages.push({type: 'info', text: `${totalImages} images found to convert.`});
                        updateLog();
                        updateProgress();
                        convertNextImage();
                    } else {
                        logMessages.push({type: 'success', text: 'No images found that need conversion.'});
                        updateLog();
                        finishConversion();
                    }
                } else {
                    handleError(response.data);
                }
            },
            error: function(xhr, status, error) {
                handleError(error);
            }
        });
    }

    $('#negarara_bulk_convert').on('click', function(e) {
        e.preventDefault();
        if (isConverting) return;

        const message = 'Are you sure you want to convert all remaining images? This action cannot be undone if the "delete original" option is checked.';
        showConfirmationModal(message, startBulkConversion);
    });

    function convertNextImage() {
        if (!isConverting) return;
        $.ajax({
            url: negararaAjax.ajaxurl,
            type: 'POST',
            data: {
                action: 'negarara_convert_single_image',
                security: negararaAjax.security,
                delete_original: $('#negarara_delete_original').is(':checked') ? 1 : 0
            },
            success: function(response) {
                if (response.success) {
                    if (response.data.log && response.data.log.length > 0) {
                        logMessages = logMessages.concat(response.data.log);
                        updateLog();
                    }
                    if (response.data.more_images) {
                        convertedImages++;
                        updateProgress();
                        convertNextImage();
                    } else {
                        finishConversion();
                    }
                } else {
                    handleError(response.data || 'Unknown error');
                }
            },
            error: function(xhr, status, error) {
                handleError(error || 'AJAX request failed');
            }
        });
    }

    function updateProgress() {
        if (totalImages > 0) {
            let percentage = Math.round((convertedImages / totalImages) * 100);
            percentage = Math.min(percentage, 100);
            progressContainer.html(`<p>Converting: ${convertedImages} / ${totalImages} (${percentage}%)</p>`);
        }
    }

    function updateLog() {
        let logHtml = logMessages.map(message => `<div class="${message.type}">${message.text}</div>`).join('');
        logContainer.html(logHtml);
        logContainer.scrollTop(logContainer[0].scrollHeight);
    }

    function finishConversion() {
        isConverting = false;
        progressContainer.html('<p style="color: var(--color-success); font-weight: bold;">Conversion complete!</p>');
        $('#negarara_bulk_convert').prop('disabled', false).text('Convert All Images');
    }

    function handleError(error) {
        console.error(error);
        logMessages.push({type: 'error', text: `An unexpected error occurred: ${JSON.stringify(error)}`});
        updateLog();
        isConverting = false;
        $('#negarara_bulk_convert').prop('disabled', false).text('Convert All Images');
        progressContainer.append('<p style="color: var(--color-error); font-weight: bold;">Conversion stopped due to an error.</p>');
    }
});