<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Negarara_Converter {
    public function __construct() {
        add_filter('wp_generate_attachment_metadata', [$this, 'convert_image_sizes_to_webp'], 20, 2);
        add_action('wp_ajax_negarara_get_convertible_image_count', [$this, 'get_convertible_image_count']);
        add_action('wp_ajax_negarara_convert_single_image', [$this, 'convert_single_image']);
    }

    public function convert_image_sizes_to_webp($metadata, $attachment_id) {
        $attachment_id = absint($attachment_id);
        $formats_to_convert = get_option('negarara_formats', ['jpeg', 'png', 'jpg']);
        $original_file_path = get_attached_file($attachment_id);

        if (!$original_file_path || !file_exists($original_file_path)) {
            return $metadata;
        }

        $file_info = pathinfo($original_file_path);
        $extension = isset($file_info['extension']) ? strtolower($file_info['extension']) : '';

        if (empty($extension) || !in_array($extension, $formats_to_convert)) {
            return $metadata;
        }

        $delete_original_file = (bool) get_option('negarara_delete_original', 0);
        $upload_dir = wp_upload_dir();

        $converted_file_path = $this->process_webp_conversion($original_file_path, $delete_original_file);

        if ($converted_file_path && $converted_file_path !== $original_file_path) {
            update_attached_file($attachment_id, $converted_file_path);
            $metadata['file'] = str_replace(trailingslashit($upload_dir['basedir']), '', $converted_file_path);
        }

        if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
            foreach ($metadata['sizes'] as $size => &$size_info) {
                $size_file_path = dirname($original_file_path) . '/' . $size_info['file'];
                if (file_exists($size_file_path) && strpos($size_info['mime-type'], 'webp') === false) {
                    $converted_size_file = $this->process_webp_conversion($size_file_path, $delete_original_file);
                    if ($converted_size_file) {
                        $size_info['file'] = basename($converted_size_file);
                        $size_info['mime-type'] = 'image/webp';
                    }
                }
            }
        }
        wp_update_post(['ID' => $attachment_id, 'post_mime_type' => 'image/webp']);
        return $metadata;
    }

    public function process_webp_conversion($file_path, $delete_original) {
        if (!file_exists($file_path)) return false;

        $file_info = pathinfo($file_path);
        $image_info = @getimagesize($file_path);

        if (!$image_info || $image_info['mime'] === 'image/gif') return false;

        $webp_file_path = $file_info['dirname'] . '/' . $file_info['filename'] . '.webp';
        $counter = 1;
        while (file_exists($webp_file_path)) {
            $webp_file_path = $file_info['dirname'] . '/' . $file_info['filename'] . '-' . $counter . '.webp';
            $counter++;
        }

        $image_editor = wp_get_image_editor($file_path);
        if (!is_wp_error($image_editor)) {
            $quality = absint(get_option('negarara_quality', 80));
            $result = $image_editor->save($webp_file_path, 'image/webp');
            if (!is_wp_error($result) && file_exists($result['path'])) {
                if ($delete_original) {
                    wp_delete_file($file_path);
                }
                return $result['path'];
            }
        }
        return false;
    }

    public function get_convertible_image_count() {
        check_ajax_referer('negarara_bulk_convert_nonce', 'security');
        if (!current_user_can('manage_options')) wp_send_json_error('Insufficient permissions');

        $selected_formats = get_option('negarara_formats', ['jpeg', 'png', 'jpg']);
        $mime_types = [];
        foreach ($selected_formats as $format) {
            if (in_array($format, ['jpeg', 'jpg'])) $mime_types['image/jpeg'] = 1;
            elseif ($format === 'png') $mime_types['image/png'] = 1;
        }
        $query = new WP_Query([
            'post_type' => 'attachment',
            'post_mime_type' => array_keys($mime_types),
            'post_status' => 'inherit',
            'posts_per_page' => -1,
            'fields' => 'ids',
        ]);
        wp_send_json_success(['count' => $query->post_count]);
    }

    public function convert_single_image() {
        check_ajax_referer('negarara_bulk_convert_nonce', 'security');
        if (!current_user_can('manage_options')) wp_send_json_error('Insufficient permissions');

        $selected_formats = get_option('negarara_formats', ['jpeg', 'png', 'jpg']);
        $mime_types = [];
        foreach ($selected_formats as $format) {
            if (in_array($format, ['jpeg', 'jpg'])) $mime_types['image/jpeg'] = 1;
            elseif ($format === 'png') $mime_types['image/png'] = 1;
        }

        $images = get_posts([
            'post_type' => 'attachment',
            'post_mime_type' => array_keys($mime_types),
            'post_status' => 'inherit',
            'posts_per_page' => 1,
            'orderby' => 'ID',
            'order' => 'DESC'
        ]);
        if (empty($images)) {
            wp_send_json_success(['more_images' => false, 'log' => [['type' => 'success', 'text' => __('All images processed.', 'negarara')]]]);
        }

        $image = $images[0];
        $attachment_id = $image->ID;
        $file_path = get_attached_file($attachment_id);
        $log[] = ['type' => 'info', 'text' => sprintf(__('Processing image: %s (ID: %d)', 'negarara'), basename($file_path), $attachment_id)];

        $metadata = wp_generate_attachment_metadata($attachment_id, $file_path);

        if (!is_wp_error($metadata) && !empty($metadata)) {
            wp_update_attachment_metadata($attachment_id, $metadata);
            $log[] = ['type' => 'success', 'text' => sprintf(__('Successfully converted image ID %d.', 'negarara'), $attachment_id)];
        } else {
            $error_message = is_wp_error($metadata) ? $metadata->get_error_message() : 'Unknown error';
            $log[] = ['type' => 'error', 'text' => sprintf(__('Failed to convert image ID %d. Reason: %s', 'negarara'), $attachment_id, $error_message)];
        }

        $remaining_images = get_posts(['post_type' => 'attachment', 'post_mime_type' => array_keys($mime_types), 'posts_per_page' => 1, 'fields' => 'ids']);
        wp_send_json_success(['more_images' => !empty($remaining_images), 'log' => $log]);
    }

    public static function get_server_support_status() {
        $support = [];
        if (extension_loaded('gd')) {
            $gd_info = gd_info();
            $support['gd'] = ['status' => (isset($gd_info['WebP Support']) && $gd_info['WebP Support']), 'message' => 'GD extension ' . ((isset($gd_info['WebP Support']) && $gd_info['WebP Support']) ? 'supports WebP.' : 'does not support WebP.')];
        } else {
            $support['gd'] = ['status' => false, 'message' => 'GD extension is not installed.'];
        }
        if (extension_loaded('imagick')) {
            $imagick = new Imagick();
            $support['imagick'] = ['status' => in_array('WEBP', $imagick->queryFormats()), 'message' => 'Imagick extension ' . (in_array('WEBP', $imagick->queryFormats()) ? 'supports WebP.' : 'does not support WebP.')];
        } else {
            $support['imagick'] = ['status' => false, 'message' => 'Imagick extension is not installed.'];
        }
        return $support;
    }
}