<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Negarara_Admin {
    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_init', [$this, 'settings_init']);
    }

    public function add_admin_menu() {
        add_options_page(
            __('Negarara Settings', 'negarara'),
            __('Negarara', 'negarara'),
            'manage_options',
            'negarara',
            [$this, 'settings_page']
        );
    }

    public function enqueue_scripts($hook) {
        if ($hook !== 'settings_page_negarara') {
            return;
        }
        wp_enqueue_script('negarara-settings-script', NEAGARARA_DIR_URL . 'js/panel.js', ['jquery'], '3.0', true);
        wp_localize_script('negarara-settings-script', 'negararaAjax', [
            'ajaxurl'  => admin_url('admin-ajax.php'),
            'security' => wp_create_nonce('negarara_bulk_convert_nonce'),
        ]);
        wp_enqueue_style('negarara-admin-style', NEAGARARA_DIR_URL . 'css/style.css', [], '3.0');
    }

    public function settings_init() {
        register_setting('negarara_settings_group', 'negarara_quality', 'absint');
        register_setting('negarara_settings_group', 'negarara_delete_original', 'absint');
        register_setting('negarara_settings_group', 'negarara_formats', [$this, 'sanitize_formats']);

        if (get_option('negarara_delete_original') === false) {
            update_option('negarara_delete_original', 0);
        }
        if (get_option('negarara_formats') === false) {
            update_option('negarara_formats', ['jpeg', 'png', 'jpg']);
        }
    }

    public function sanitize_formats($input) {
        if (is_array($input)) {
            return array_map('sanitize_text_field', $input);
        }
        return [];
    }

    public function settings_page() {
        ?>
        <div class="mwtc_admin_settings_wrapper <?php echo esc_attr(is_rtl() ? 'mwtc-rtl' : 'mwtc-ltr'); ?>">
            <div class="mwtc_sidebar_wrapper">
                <div class="sidebar">
                    <div class="mw_logo_section">
                        <div class="mw_logo">
                            <img src="<?php echo esc_url(NEAGARARA_DIR_URL . 'img/logo.svg'); ?>" alt="<?php esc_attr_e('Logo', 'negarara'); ?>">
                        </div>
                        <div class="mw_hello">
                            <span><?php printf('%s %s', esc_html__('Welcome to NegarAra', 'negarara'), '👋'); ?></span>
                        </div>
                    </div>
                    <div class="mw_menu_section">
                        <ul>
                            <li class="active">
                                <a href="#">
                                    <span class="menu-icon" tab="note"></span>
                                    <span class="menu-name"><?php esc_html_e('General', 'negarara'); ?></span>
                                </a>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>

            <div class="content_wrapper">
                <div class="content">
                    <form method="post" action="options.php" class="mp_options_panel" id="negarara-option-panel-form">
                        <?php
                        settings_fields('negarara_settings_group');
                        do_settings_sections('negarara_settings');
                        ?>

                        <div class="option_section">
                            <h3 class="option_section_title"><?php esc_html_e('Server Status', 'negarara'); ?></h3>
                            <div class="option_field option_col_field">
                                <?php $server_status = Negarara_Converter::get_server_support_status(); ?>
                                <ul class="server-status-list">
                                    <?php foreach ($server_status as $extension => $status): ?>
                                        <li class="<?php echo $status['status'] ? 'status-success' : 'status-error'; ?>">
                                            <span class="status-icon"></span> <span class="status-label"><?php echo esc_html(strtoupper($extension)); ?>:</span>
                                            <span class="status-message"><?php echo esc_html($status['message']); ?></span>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>

                        <div class="option_section">
                            <h3 class="option_section_title"><?php esc_html_e('Settings', 'negarara'); ?></h3>
                            <div class="option_field">
                                <?php $quality = get_option('negarara_quality', 80); ?>
                                <label for="negarara_quality"><?php esc_html_e('WebP Image Quality', 'negarara'); ?></label>
                                <input type="range" id="negarara_quality" class="mwpl_range" name="negarara_quality" min="10" max="100" value="<?php echo esc_attr($quality); ?>">
                                <span class="range_value"><?php echo sprintf('( %s )', esc_html($quality)); ?></span>
                            </div>
                        </div>

                        <div class="option_section">
                            <h3 class="option_section_title"><?php esc_html_e('Advanced Settings', 'negarara'); ?></h3>
                            <div class="option_field option_col_field">
                                <div class="toggle-wrapper">
                                    <label><?php esc_html_e('Convert JPEG/JPG', 'negarara'); ?></label>
                                    <div class="toggle-switch">
                                        <input type="checkbox" id="negarara_format_jpeg" name="negarara_formats[]" value="jpeg" <?php checked(in_array('jpeg', get_option('negarara_formats', ['jpeg', 'png', 'jpg']))); ?>>
                                        <label for="negarara_format_jpeg"></label>
                                    </div>
                                </div>
                                <div class="toggle-wrapper">
                                    <label><?php esc_html_e('Convert PNG', 'negarara'); ?></label>
                                    <div class="toggle-switch">
                                        <input type="checkbox" id="negarara_format_png" name="negarara_formats[]" value="png" <?php checked(in_array('png', get_option('negarara_formats', ['jpeg', 'png', 'jpg']))); ?>>
                                        <label for="negarara_format_png"></label>
                                    </div>
                                </div>
                                <div class="toggle-wrapper">
                                    <label><?php esc_html_e('Delete original files after conversion', 'negarara'); ?></label>
                                    <div class="toggle-switch">
                                        <input type="checkbox" id="negarara_delete_original" name="negarara_delete_original" value="1" <?php checked(get_option('negarara_delete_original', 0)); ?>>
                                        <label for="negarara_delete_original"></label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="option_section">
                            <h3 class="option_section_title"><?php esc_html_e('Actions', 'negarara'); ?></h3>
                            <div class="option_field">
                                <input type="submit" class="button button-primary button-full" value="<?php esc_attr_e('Save Changes', 'negarara'); ?>" />
                            </div>
                        </div>

                        <div class="option_section">
                            <h3 class="option_section_title"><?php esc_html_e('Bulk Conversion', 'negarara'); ?></h3>
                            <p class="option_section_description"><?php esc_html_e('Convert all existing original images to WebP.', 'negarara'); ?></p>
                            <button id="negarara_bulk_convert" class="button button-primary button-full"><?php esc_html_e('Convert All Images', 'negarara'); ?></button>
                            <div id="negarara_conversion_progress" style="margin-top:15px;"></div>
                            <div id="negarara_conversion_log" style="display: none;"></div>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div id="negarara-modal-overlay" style="display: none;">
            <div id="negarara-modal">
                <h3 id="negarara-modal-title"><?php esc_html_e('Confirm Action', 'negarara'); ?></h3>
                <p id="negarara-modal-message"></p>
                <div id="negarara-modal-actions">
                    <button id="negarara-modal-cancel" class="button"><?php esc_html_e('Cancel', 'negarara'); ?></button>
                    <button id="negarara-modal-confirm" class="button button-primary"><?php esc_html_e('Proceed', 'negarara'); ?></button>
                </div>
            </div>
        </div>
        <?php
    }
}